# standard-engine [![Tests CI][ci-image]][ci-url] [![npm][npm-image]][npm-url] [![downloads][downloads-image]][downloads-url] [![javascript style guide][standard-image]][standard-url]

[ci-image]: https://github.com/standard/standard-engine/workflows/tests/badge.svg?branch=master
[ci-url]: https://github.com/standard/standard-engine/actions?query=workflow%3A%22tests%22
[npm-image]: https://img.shields.io/npm/v/standard-engine.svg
[npm-url]: https://npmjs.org/package/standard-engine
[downloads-image]: https://img.shields.io/npm/dm/standard-engine.svg
[downloads-url]: https://npmjs.org/package/standard-engine
[standard-image]: https://img.shields.io/badge/code_style-standard-brightgreen.svg
[standard-url]: https://standardjs.com

## Overview

Wrap your own eslint rules in a easy-to-use command line tool and/or a JS module.

## Install

```sh
npm install standard-engine
```

## Who is using `standard-engine`?

Here is a list of packages using `standard-engine`. Dive into them for ideas!

- [standard](https://standardjs.com) - JavaScript Standard Style.
- [semistandard](https://github.com/standard/semistandard) - It's `standard` with semicolons sprinkled on top.
- [standardx](https://github.com/standard/standardx) - JavaScript Standard Style with custom tweaks
- [happiness](https://github.com/JedWatson/happiness) - JavaScript Happiness Style (semicolons and tabs)
- [doublestandard](https://github.com/flet/doublestandard) - Require TWO semicolons at the end of every line!
- [strict-standard](https://github.com/denis-sokolov/strict-standard) - Standard Style with strict error checking.
- [standard-own](https://github.com/o2team/standard-own) - Standard configurable.
- [ts-standard](https://github.com/standard/ts-standard) - TypeScript Standard Style.

Did you make your own? Create a pull request and we will add it to the README!

## Usage

Create the files below and fill in your own values for `options.js`.

### `index.js`

```js
// programmatic usage
const { StandardEngine } = require('standard-engine')
const opts = require('./options.js')
module.exports = new StandardEngine(opts)
```

### `cli.js`

```js
#!/usr/bin/env node

const opts = require('./options.js')

require('standard-engine').cli(opts)
```

### `options.js`

```js
const eslint = require('eslint')
const path = require('path')
const pkg = require('./package.json')

/** @type {import('standard-engine').StandardEngineOptions} **/
module.exports = {
  // homepage, version and bugs pulled from package.json
  version: pkg.version,
  homepage: pkg.homepage,
  bugs: pkg.bugs.url,
  eslint, // pass any version of eslint >= 7.0.0
  cmd: 'pocketlint', // should match the "bin" key in your package.json
  tagline: 'Live by your own standards!', // displayed in output --help
  eslintConfig: {
    overrideConfigFile: path.join(__dirname, 'eslintrc.json')
  }
}
```

Additionally an optional `resolveEslintConfig()` function can be provided. See below for details.

### `eslintrc.json`

Put all your .eslintrc rules in this file. A good practice is to create an  [ESLint Shareable Config](http://eslint.org/docs/developer-guide/shareable-configs) and extend it, but its not required:

```js
{
  // pretend that the package eslint-config-pocketlint exists!
  "extends": ["pocketlint"]
}
```

Take a look at [eslint-config-standard](https://github.com/feross/eslint-config-standard) as an example, or if you want to extend/mutate `standard`, see [eslint-config-semistandard](https://github.com/flet/eslint-config-semistandard).

## Editor Integrations

Integrations and plugins should recognize the `standard-engine` tag in a
`package.json` file. This allows end users to specify an arbitrary `standard-engine`
compatible linter that the plugin should use. The `standard-engine` tag can be a
string of the package:

```json
{
  "standard-engine": "pocketlint"
}
```

or an object with a `name` value of the package:

```json
{
  "standard-engine": {
    "name": "pocketlint"
  }
}
```

### Atom

[linter-js-standard-engine][atom-plugin] is an Atom plugin that supports some of
the more popular standard-engine implementations out of the box. It detects them
by scanning through the dependencies of the project that you are editing.
You can use it with any other implementation through configuration in the
projects `package.json` file.

[atom-plugin]: https://github.com/gustavnikolaj/linter-js-standard-engine

## Engine Features

### Extensions

The extensions `.js`, `.jsx`, `.mjs`, and `.cjs` are linted by default. If you
pass directory paths to the `standardEngine.lintFiles()` method,
`standard-engine` checks the files in those directories that have the given
extensions.

For example, when passing the `src/` directory and the `extensions` option is
`['.js', '.jsx']`, `standard-engine` will lint `*.js` and `*.jsx` files in
`src/`.

You can disable these default ignores by setting the `noDefaultExensions` option to `true`.

### Ignoring Files

The paths `node_modules/**`, `*.min.js`, `coverage/**`, hidden files/folders
(beginning with `.`), and all patterns in a project's root `.gitignore` file are
automatically ignored.

Sometimes you need to ignore additional folders or specific minfied files. To do that, add
a `ignore` property to `package.json`:

```js
"pocketlint": { // this key should equal the value of cmd in options.js
  "ignore": [
    "**/out/",
    "/lib/select2/",
    "/lib/ckeditor/",
    "tmp.js"
  ]
}
```

Some files are ignored by default:

```js
const DEFAULT_IGNORE = [
  '*.min.js',
  'coverage/',
  'node_modules/',
  'vendor/'
]
```

You can disable these default ignores by setting the `noDefaultIgnore` option to `true`.

### Hiding Warnings

Since `standard-engine` uses [`eslint`](http://eslint.org/) under-the-hood, you can hide warnings as you normally would if you used `eslint` directly.

Disable **all rules** on a specific line:

```js
file = 'I know what I am doing' // eslint-disable-line
```

Or, disable **only** the `"no-use-before-define"` rule:

```js
file = 'I know what I am doing' // eslint-disable-line no-use-before-define
```

Or, disable the `"no-use-before-define"` rule for **multiple lines**:

```js
/*eslint-disable no-use-before-define */
// offending code here...
// offending code here...
// offending code here...
/*eslint-enable no-use-before-define */
```

### Defining Globals in a project's `package.json`

`standard-engine` will also look in a project's `package.json` and respect any global variables defined like so:

```js
{
  "pocketlint": { // this key should equal the value of cmd in options.js
    "globals": [ // can be a string or an array of strings
      "myVar1",
      "myVar2"
    ]
  }
}
```

You may use `global` as an alias for `globals` (just don't specify both).

### Loading ESLint plugins in a project's `package.json`

Additional ESLint plugins can be specified like so:

```js
{
  "pocketlint": { // this key should equal the value of cmd in options.js
    "plugins": [ // can be a string or an array of strings
      "flowtype"
    ]
  }
}
```

You may use `plugin` as an alias for `plugins` (just don't specify both). Plugins must be installed (example: `npm install eslint-plugin-flowtype` or globally: `npm install eslint-plugin-flowtype -g`).

### Loading additional environments in a project's `package.json`

Additional environments can be specified like so:

```js
{
  "pocketlint": { // this key should equal the value of cmd in options.js
    "envs": [ "browser", "mocha" ]
  }
}
```

`envs` can be a string, an array of strings, or an object. In the latter case the keys are used as the environment name, but falsy values mean the environment is not actually loaded. You cannot unload environments by setting a falsy value.

You may use `env` as an alias for `envs` (just don't specify both).

### Custom JS parsers for bleeding-edge ES6 or ES7 support?

`standard-engine` supports custom JS parsers. To use a custom parser, install it from npm
(example: `npm install babel-eslint`) and add this to your `package.json`:

```js
{
  "pocketlint": { // this key should equal the value of cmd in your options.js
    "parser": "babel-eslint"
  }
}
```

If you're using your custom linter globally (you installed it with `-g`), then you also need to
install `babel-eslint` globally with `npm install babel-eslint -g`.

### Custom options

You can provide a `resolveEslintConfig()` function in the `options.js` exports:

```js
const eslint = require('eslint')
const path = require('path')
const pkg = require('./package.json')

module.exports = {
  // homepage, version and bugs pulled from package.json
  version: pkg.version,
  homepage: pkg.homepage,
  bugs: pkg.bugs.url,
  eslint, // pass any version of eslint >= 7.0.0
  cmd: 'pocketlint', // should match the "bin" key in your package.json
  tagline: 'Live by your own standards!', // displayed in output --help
  eslintConfig: {
    overrideConfigFile: path.join(__dirname, 'eslintrc.json')
  },
  resolveEslintConfig: function (eslintConfig, opts, packageOpts, rootDir) {
    // provide implementation here, then return the eslintConfig object (or a new one)
    return eslintConfig
  }
}
```

This function is called with the current ESLint config (the options passed to the [`ESLint`](https://eslint.org/docs/developer-guide/nodejs-api#-new-eslintoptions) constructor), the options object (`opts`), any options extracted from the project's `package.json` (`packageOpts`), and the directory that contained that `package.json` file (`rootDir`, equivalent to `opts.cwd` if no file was found).

Modify and return `eslintConfig`, or return a new object with the eslint config to be used.

## API Usage

### `async engine.lintText(text, [opts])`

Lint the provided source `text` to enforce your defined style. An `opts` object may
be provided:

```js
{
  // unique to lintText
  filename: '',         // path of file containing the text being linted

  // common to lintText and lintFiles
  cwd: '',              // current working directory (default: process.cwd())
  fix: false,           // automatically fix problems
  extensions: [],       // file extensions to lint (has sane defaults)
  globals: [],          // custom global variables to declare
  plugins: [],          // custom eslint plugins
  envs: [],             // custom eslint environment
  parser: '',           // custom js parser (e.g. babel-eslint)
  usePackageJson: true, // use options from nearest package.json?
  useGitIgnore: true    // use file ignore patterns from .gitignore?
}
```

All options are optional, though some ESLint plugins require the `filename` option.

Additional options may be loaded from a `package.json` if it's found for the current working directory. See below for further details.

Returns a `Promise` resolving to the `results` or rejected with an `Error`.

The `results` object will contain the following properties:

```js
const results = {
  results: [
    {
      filePath: '',
      messages: [
        { ruleId: '', message: '', line: 0, column: 0 }
      ],
      errorCount: 0,
      warningCount: 0,
      output: '' // fixed source code (only present with {fix: true} option)
    }
  ],
  errorCount: 0,
  warningCount: 0
}
```

### `async engine.lintFiles(files, [opts])`

Lint the provided `files` globs. An `opts` object may be provided:

```js
{
  // unique to lintFiles
  ignore: [],           // file globs to ignore (has sane defaults)

  // common to lintText and lintFiles
  cwd: '',              // current working directory (default: process.cwd())
  fix: false,           // automatically fix problems
  extensions: [],       // file extensions to lint (has sane defaults)
  globals: [],          // custom global variables to declare
  plugins: [],          // custom eslint plugins
  envs: [],             // custom eslint environment
  parser: '',           // custom js parser (e.g. babel-eslint)
  usePackageJson: true, // use options from nearest package.json?
  useGitIgnore: true    // use file ignore patterns from .gitignore?
}
```

Additional options may be loaded from a `package.json` if it's found for the current working directory. See below for further details.

Both `ignore` and `files` patterns are resolved relative to the current working directory.

Returns a `Promise` resolving to the `results` or rejected with an `Error` (same as above).

**NOTE: There is no synchronous version of `engine.lintFiles()`.**

### Full set of `opts`

This is the full set of options accepted by the above APIs. Not all options make sense for each API, for example `ignore` is not used with `lintText()`, and `filename` is not used with `lintFiles()`.

```js
{
  ignore: [],   // file patterns to ignore (has sane defaults)
  cwd: '',      // current working directory (default: process.cwd())
  filename: '', // path of the file containing the text being linted (optional)
  fix: false,   // automatically fix problems
  globals: [],  // custom global variables to declare
  plugins: [],  // custom eslint plugins
  envs: [],     // custom eslint environment
  parser: ''    // custom js parser (e.g. babel-eslint)
}
```

The following aliases are available:

```js
{
  global: [],  // custom global variables to declare
  plugin: [],  // custom eslint plugins
  env: [],     // custom eslint environment
}
```

Note that `globals`, `plugins` and `envs` take preference.

The `parser` option takes preference over any `parser` setting in the project's `package.json`.
