"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
exports.__esModule = true;
var isBuilt = function (module) { return module.built; };
var getId = function (module) { return module.id; };
var affectedModules = function (map, usageMap, affected, moduleId) {
    if (typeof affected[moduleId] !== 'undefined') {
        // module was already inspected, stop here otherwise we get into endless recursion
        return;
    }
    // module is identified as affected by this function call
    var module = map[moduleId];
    affected[module.id] = module; // eslint-disable-line no-param-reassign
    // next we need to mark all usages aka parents also as affected
    var usages = usageMap[module.id];
    if (typeof usages !== 'undefined') {
        var ids = Object.keys(usages);
        ids.forEach(function (id) { return affectedModules(map, usageMap, affected, id); });
    }
};
/**
 * Builds a map where all modules are indexed by it's id
 * {
 *   [moduleId]: Module
 * }
 */
var buildModuleMap = function (modules) {
    var moduleMap = modules.reduce(function (memo, module) {
        var _a;
        return (__assign(__assign({}, memo), (_a = {}, _a[module.id] = module, _a)));
    }, {});
    return moduleMap;
};
/**
 * Builds a map with all modules that are used in other modules (child -> parent relation)
 *
 * {
 *  [childModuleId]: {
 *    [parentModuleId]: ParentModule
 *  }
 * }
 *
 * @param modules Array<number>
 * @return ModuleUsageMap
 */
var buildModuleUsageMap = function (chunks, modules) {
    // build a map of all modules with their parent
    // {
    //    [childModuleId]: {
    //      [parentModuleId]: ParentModule
    //    }
    // }
    //
    var moduleUsageMap = modules.reduce(function (memo, module) {
        module.dependencies.forEach(function (dependency) {
            var dependentModule = dependency.module;
            if (!dependentModule) {
                return;
            }
            if (typeof memo[dependentModule.id] === 'undefined') {
                memo[dependentModule.id] = {}; // eslint-disable-line no-param-reassign
            }
            memo[dependentModule.id][module.id] = module; // eslint-disable-line no-param-reassign
        });
        return memo;
    }, {});
    // build a map of all chunks with their modules
    // {
    //    [chunkId]: {
    //      [moduleId]: Module
    //    }
    // }
    var chunkModuleMap = chunks.reduce(function (memo, chunk) {
        // build chunk map first to get also empty chunks (without modules)
        memo[chunk.id] = {}; // eslint-disable-line no-param-reassign
        return memo;
    }, {});
    modules.reduce(function (memo, module) {
        module.getChunks().forEach(function (chunk) {
            memo[chunk.id][module.id] = module; // eslint-disable-line no-param-reassign
        });
        return memo;
    }, chunkModuleMap);
    // detect modules with code split points (e.g. require.ensure) and enhance moduleUsageMap with that information
    modules.forEach(function (module) {
        module.blocks // chunkGroup can be invalid in in some cases
            .filter(function (block) { return block.chunkGroup != null; })
            .forEach(function (block) {
            // loop through all generated chunks by this module
            block.chunkGroup.chunks.map(getId).forEach(function (chunkId) {
                // and mark all modules of this chunk as a direct dependency of the original module
                Object.values(chunkModuleMap[chunkId]).forEach(function (childModule) {
                    if (typeof moduleUsageMap[childModule.id] === 'undefined') {
                        moduleUsageMap[childModule.id] = {};
                    }
                    moduleUsageMap[childModule.id][module.id] = module;
                });
            });
        });
    });
    return moduleUsageMap;
};
/**
 * Builds a list with ids of all affected modules in the following way:
 *  - affected directly by a file change
 *  - affected indirectly by a change of it's dependencies and so on
 *
 * @param chunks Array<Chunk>
 * @param modules Array<Module>
 * @return {Array.<number>}
 */
function getAffectedModuleIds(chunks, modules) {
    var moduleMap = buildModuleMap(modules);
    var moduleUsageMap = buildModuleUsageMap(chunks, modules);
    var builtModules = modules.filter(isBuilt);
    var affectedMap = {};
    builtModules.forEach(function (module) {
        return affectedModules(moduleMap, moduleUsageMap, affectedMap, module.id);
    });
    return Object.values(affectedMap).map(getId);
}
exports["default"] = getAffectedModuleIds;
//# sourceMappingURL=webpack4GetAffectedModuleIds.js.map