"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
exports.__esModule = true;
exports.mochaCliOptionArgs = void 0;
var lodash_1 = require("lodash");
var mergeMochaConfigWithOptions_1 = require("./mergeMochaConfigWithOptions");
var utils_1 = __importDefault(require("../../utils"));
/**
 * Keep track of which keys are not applicable to the Mocha constructor here.
 */
exports.mochaCliOptionArgs = [
    'config',
    'exit',
    'extension',
    'file',
    'files',
    'ignore',
    'invert',
    'list-interfaces',
    'list-reporters',
    'package',
    'recursive',
    'require',
    'sort',
    'watch',
    'watch-files',
    'watch-ignore'
];
/**
 * These arguments require some processing in order to be translated into an
 *   `MochaOptions` object to be used by Mochapack when initializing an
 *   instance of `Mocha`
 */
var argsThatDoNotDirectlyTranslateToMochaOptions = [
    'check-leaks',
    'color',
    'diff',
    'full-trace',
    'fgrep',
    'global',
    'grep',
    'no-colors',
    'reporter-option'
];
/**
 * Ensures that any options that are provided as strings but expected as
 *   numbers are converted to numbers in the output
 *
 * @param camelizedArgs Arguments that have been parsed and their keys have
 *   camelized
 */
var ensureNumericOptionsAreNumbers = function (camelizedArgs) {
    var optionsThatShouldBeNumbers = ['slow', 'timeout'];
    var output = lodash_1.cloneDeep(camelizedArgs);
    optionsThatShouldBeNumbers.forEach(function (optionName) {
        if (output[optionName])
            output[optionName] = parseInt(output[optionName], 10);
    });
    return output;
};
var colorsShouldBeUsed = function (args) {
    if (args['no-colors'])
        return false;
    return !!args.color;
};
var grepToUse = function (args) {
    if (args.grep)
        return args.grep;
    if (args.fgrep)
        return args.fgrep;
    return undefined;
};
/**
 * Translates camelized arguments into a `MochaOptions` object that can be
 *   directly provided to a Mocha initializer
 */
var translateObjectIntoMochaOptions = function (args) {
    var oneToOnes = lodash_1.omit(args, argsThatDoNotDirectlyTranslateToMochaOptions);
    var ignoreLeaks = !args['check-leaks'];
    var useColors = colorsShouldBeUsed(args);
    var hideDiff = !args.diff;
    var fullStackTrace = args['full-trace'];
    var globals = args.global;
    var grep = grepToUse(args);
    var reporterOptions = args['reporter-option'];
    var options = lodash_1.merge({}, utils_1["default"].camelizeKeys(oneToOnes), {
        ignoreLeaks: ignoreLeaks,
        useColors: useColors,
        hideDiff: hideDiff,
        fullStackTrace: fullStackTrace,
        globals: globals,
        grep: grep,
        reporterOptions: reporterOptions
    });
    return ensureNumericOptionsAreNumbers(options);
};
/**
 * Extracts applicable options for Mocha constructor given a `ParsedMochaArgs`
 *   object
 *
 * @param parsedMochaArgs A `ParsedMochaArgs` object
 */
var extractMochaConstructorOptions = function (parsedMochaArgs) {
    var relevantArgs = lodash_1.omit(parsedMochaArgs, exports.mochaCliOptionArgs);
    var mochaOptions = translateObjectIntoMochaOptions(relevantArgs);
    var mergedOptions = mergeMochaConfigWithOptions_1.mergeMochaConfigWithConstructorOptions(mochaOptions, parsedMochaArgs.config);
    return mergedOptions;
};
/**
 * Extracts applicable options for Mocha CLI given a `ParsedMochaArgs`
 *   object
 *
 * @param parsedMochaArgs A `ParsedMochaArgs` object
 */
var extractMochaCliOptions = function (parsedMochaArgs) {
    var relevantArgs = lodash_1.pick(parsedMochaArgs, exports.mochaCliOptionArgs);
    var camelizedArgs = utils_1["default"].camelizeKeys(relevantArgs);
    var mergedOptions = mergeMochaConfigWithOptions_1.mergeMochaConfigWithCliOptions(camelizedArgs, parsedMochaArgs.config, parsedMochaArgs.opts);
    return mergedOptions;
};
/**
 * Translates incoming arguments relevant to Mocha into options to be used in
 *   constructor of Mocha instance and in CLI run by Mochapack
 *
 * @param parsedArgs Arguments that have been parsed from CLI
 */
var mochaOptionsFromParsedArgs = function (parsedArgs) { return ({
    constructor: extractMochaConstructorOptions(parsedArgs.mocha),
    cli: extractMochaCliOptions(parsedArgs.mocha)
}); };
exports["default"] = mochaOptionsFromParsedArgs;
//# sourceMappingURL=mochaOptionsFromParsedArgs.js.map