/*
 * World Calendars
 * https://github.com/alexcjohnson/world-calendars
 *
 * Batch-converted from kbwood/calendars
 * Many thanks to Keith Wood and all of the contributors to the original project!
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

﻿/* http://keith-wood.name/calendars.html
   Ethiopian calendar for jQuery v2.0.2.
   Written by Keith Wood (wood.keith{at}optusnet.com.au) February 2010.
   Available under the MIT (http://keith-wood.name/licence.html) license. 
   Please attribute the author if you use it. */

var main = require('../main');
var assign = require('object-assign');


/** Implementation of the Ethiopian calendar.
    See <a href="http://en.wikipedia.org/wiki/Ethiopian_calendar">http://en.wikipedia.org/wiki/Ethiopian_calendar</a>.
    See also Calendrical Calculations: The Millennium Edition
    (<a href="http://emr.cs.iit.edu/home/reingold/calendar-book/index.shtml">http://emr.cs.iit.edu/home/reingold/calendar-book/index.shtml</a>).
    @class EthiopianCalendar
    @param [language=''] {string} The language code (default English) for localisation. */
function EthiopianCalendar(language) {
    this.local = this.regionalOptions[language || ''] || this.regionalOptions[''];
}

EthiopianCalendar.prototype = new main.baseCalendar;

assign(EthiopianCalendar.prototype, {
    /** The calendar name.
        @memberof EthiopianCalendar */
    name: 'Ethiopian',
    /** Julian date of start of Ethiopian epoch: 27 August 8 CE (Gregorian).
        @memberof EthiopianCalendar */
    jdEpoch: 1724220.5,
    /** Days per month in a common year.
        @memberof EthiopianCalendar */
    daysPerMonth: [30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 5],
    /** <code>true</code> if has a year zero, <code>false</code> if not.
        @memberof EthiopianCalendar */
    hasYearZero: false,
    /** The minimum month number.
        @memberof EthiopianCalendar */
    minMonth: 1,
    /** The first month in the year.
        @memberof EthiopianCalendar */
    firstMonth: 1,
    /** The minimum day number.
        @memberof EthiopianCalendar */
    minDay: 1,

    /** Localisations for the plugin.
        Entries are objects indexed by the language code ('' being the default US/English).
        Each object has the following attributes.
        @memberof EthiopianCalendar
        @property name {string} The calendar name.
        @property epochs {string[]} The epoch names.
        @property monthNames {string[]} The long names of the months of the year.
        @property monthNamesShort {string[]} The short names of the months of the year.
        @property dayNames {string[]} The long names of the days of the week.
        @property dayNamesShort {string[]} The short names of the days of the week.
        @property dayNamesMin {string[]} The minimal names of the days of the week.
        @property dateFormat {string} The date format for this calendar.
                See the options on <a href="BaseCalendar.html#formatDate"><code>formatDate</code></a> for details.
        @property firstDay {number} The number of the first day of the week, starting at 0.
        @property isRTL {number} <code>true</code> if this localisation reads right-to-left. */
    regionalOptions: { // Localisations
        '': {
            name: 'Ethiopian',
            epochs: ['BEE', 'EE'],
            monthNames: ['Meskerem', 'Tikemet', 'Hidar', 'Tahesas', 'Tir', 'Yekatit',
            'Megabit', 'Miazia', 'Genbot', 'Sene', 'Hamle', 'Nehase', 'Pagume'],
            monthNamesShort: ['Mes', 'Tik', 'Hid', 'Tah', 'Tir', 'Yek',
            'Meg', 'Mia', 'Gen', 'Sen', 'Ham', 'Neh', 'Pag'],
            dayNames: ['Ehud', 'Segno', 'Maksegno', 'Irob', 'Hamus', 'Arb', 'Kidame'],
            dayNamesShort: ['Ehu', 'Seg', 'Mak', 'Iro', 'Ham', 'Arb', 'Kid'],
            dayNamesMin: ['Eh', 'Se', 'Ma', 'Ir', 'Ha', 'Ar', 'Ki'],
            digits: null,
            dateFormat: 'dd/mm/yyyy',
            firstDay: 0,
            isRTL: false
        }
    },

    /** Determine whether this date is in a leap year.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to examine or the year to examine.
        @return {boolean} <code>true</code> if this is a leap year, <code>false</code> if not.
        @throws Error if an invalid year or a different calendar used. */
    leapYear: function(year) {
        var date = this._validate(year, this.minMonth, this.minDay, main.local.invalidYear);
        var year = date.year() + (date.year() < 0 ? 1 : 0); // No year zero
        return year % 4 === 3 || year % 4 === -1;
    },

    /** Retrieve the number of months in a year.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to examine or the year to examine.
        @return {number} The number of months.
        @throws Error if an invalid year or a different calendar used. */
    monthsInYear: function(year) {
        this._validate(year, this.minMonth, this.minDay,
            main.local.invalidYear || main.regionalOptions[''].invalidYear);
        return 13;
    },

    /** Determine the week of the year for a date.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to examine or the year to examine.
        @param [month] {number} The month to examine.
        @param [day] {number} The day to examine.
        @return {number} The week of the year.
        @throws Error if an invalid date or a different calendar used. */
    weekOfYear: function(year, month, day) {
        // Find Sunday of this week starting on Sunday
        var checkDate = this.newDate(year, month, day);
        checkDate.add(-checkDate.dayOfWeek(), 'd');
        return Math.floor((checkDate.dayOfYear() - 1) / 7) + 1;
    },

    /** Retrieve the number of days in a month.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to examine or the year of the month.
        @param [month] {number} The month.
        @return {number} The number of days in this month.
        @throws Error if an invalid month/year or a different calendar used. */
    daysInMonth: function(year, month) {
        var date = this._validate(year, month, this.minDay, main.local.invalidMonth);
        return this.daysPerMonth[date.month() - 1] +
            (date.month() === 13 && this.leapYear(date.year()) ? 1 : 0);
    },

    /** Determine whether this date is a week day.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to examine or the year to examine.
        @param [month] {number} The month to examine.
        @param [day] {number} The day to examine.
        @return {boolean} <code>true</code> if a week day, <code>false</code> if not.
        @throws Error if an invalid date or a different calendar used. */
    weekDay: function(year, month, day) {
        return (this.dayOfWeek(year, month, day) || 7) < 6;
    },

    /** Retrieve the Julian date equivalent for this date,
        i.e. days since January 1, 4713 BCE Greenwich noon.
        @memberof EthiopianCalendar
        @param year {CDate|number} The date to convert or the year to convert.
        @param [month] {number} The month to convert.
        @param [day] {number} The day to convert.
        @return {number} The equivalent Julian date.
        @throws Error if an invalid date or a different calendar used. */
    toJD: function(year, month, day) {
        var date = this._validate(year, month, day, main.local.invalidDate);
        year = date.year();
        if (year < 0) { year++; } // No year zero
        return date.day() + (date.month() - 1) * 30 +
            (year - 1) * 365 + Math.floor(year / 4) + this.jdEpoch - 1;
    },

    /** Create a new date from a Julian date.
        @memberof EthiopianCalendar
        @param jd {number} the Julian date to convert.
        @return {CDate} the equivalent date. */
    fromJD: function(jd) {
        var c = Math.floor(jd) + 0.5 - this.jdEpoch;
        var year = Math.floor((c - Math.floor((c + 366) / 1461)) / 365) + 1;
        if (year <= 0) { year--; } // No year zero
        c = Math.floor(jd) + 0.5 - this.newDate(year, 1, 1).toJD();
        var month = Math.floor(c / 30) + 1;
        var day = c - (month - 1) * 30 + 1;
        return this.newDate(year, month, day);
    }
});

// Ethiopian calendar implementation
main.calendars.ethiopian = EthiopianCalendar;

