"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
exports.__esModule = true;
var chalk_1 = __importDefault(require("chalk"));
var createStatsFormatter_1 = __importDefault(require("../webpack/util/createStatsFormatter"));
var constants_1 = require("../util/constants");
var log = function () {
    var args = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        args[_i] = arguments[_i];
    }
    console.log.apply(console, args); // eslint-disable-line no-console
    console.log(); // eslint-disable-line no-console
};
var formatTitleInfo = function (title) { return chalk_1["default"].inverse('', title, ''); };
var formatTitleWarn = function (title) { return chalk_1["default"].black.bgYellow('', title, ''); };
var formatTitleError = function (title) { return chalk_1["default"].white.bold.bgRed('', title, ''); };
var Reporter = /** @class */ (function () {
    function Reporter(options) {
        var _this = this;
        this.onUncaughtException = function (err) {
            log(formatTitleError('UNCAUGHT EXCEPTION'), 'Exception occurred after running tests');
            log(err.stack);
        };
        this.onLoadingException = function (err) {
            log(formatTitleError('RUNTIME EXCEPTION'), 'Exception occurred while loading your tests');
            log(err.stack);
        };
        this.onWebpackStart = function () {
            _this.clearTerminal();
            if (_this.added.length > 0) {
                _this.logInfo(formatTitleInfo('MOCHA'), 'The following test entry files were added:');
                _this.logInfo(_this.added.map(function (f) { return "+ " + f; }).join('\n'));
            }
            if (_this.removed.length > 0) {
                _this.logInfo(formatTitleInfo('MOCHA'), 'The following test entry files were removed:');
                _this.logInfo(_this.removed.map(function (f) { return "- " + f; }).join('\n'));
            }
            _this.logInfo(formatTitleInfo('WEBPACK'), 'Compiling...');
            _this.added.length = 0;
            _this.removed.length = 0;
        };
        this.onWebpackReady = function (err, stats) {
            _this.clearTerminal();
            if (stats != null) {
                var _a = _this.formatStats(stats), errors = _a.errors, warnings = _a.warnings;
                if (errors.length === 0 && warnings.length === 0) {
                    var startTime = stats.startTime, endTime = stats.endTime;
                    var compileTime = endTime - startTime;
                    _this.logInfo(formatTitleInfo('WEBPACK'), "Compiled successfully in " + chalk_1["default"].green(compileTime + "ms"));
                    return;
                }
                if (errors.length > 0) {
                    Reporter.displayErrors('error', errors);
                    return;
                }
                if (warnings.length > 0) {
                    Reporter.displayErrors('warning', warnings);
                }
            }
            else {
                Reporter.displayErrors('error', [err]);
            }
        };
        this.onMochaStart = function () {
            _this.logInfo(formatTitleInfo('MOCHA'), 'Testing...');
        };
        this.onMochaAbort = function () {
            _this.logInfo(formatTitleInfo('MOCHA'), 'Tests aborted');
        };
        this.onMochaReady = function (failures) {
            if (failures === 0) {
                _this.logInfo(formatTitleInfo('MOCHA'), "Tests completed " + chalk_1["default"].green('successfully'));
            }
            else {
                _this.logInfo(formatTitleInfo('MOCHA'), "Tests completed with " + chalk_1["default"].red(failures + " failure(s)"));
            }
        };
        this.onEntryAdded = function (file) {
            _this.added.push(file);
        };
        this.onEntryRemoved = function (file) {
            _this.removed.push(file);
        };
        var cwd = options.cwd, eventEmitter = options.eventEmitter;
        this.options = options;
        this.added = [];
        this.removed = [];
        this.formatStats = createStatsFormatter_1["default"](cwd);
        eventEmitter.on(constants_1.UNCAUGHT_EXCEPTION_EVENT, this.onUncaughtException);
        eventEmitter.on(constants_1.EXCEPTION_EVENT, this.onLoadingException);
        eventEmitter.on(constants_1.WEBPACK_START_EVENT, this.onWebpackStart);
        eventEmitter.on(constants_1.WEBPACK_READY_EVENT, this.onWebpackReady);
        eventEmitter.on(constants_1.MOCHA_BEGIN_EVENT, this.onMochaStart);
        eventEmitter.on(constants_1.MOCHA_ABORTED_EVENT, this.onMochaAbort);
        eventEmitter.on(constants_1.MOCHA_FINISHED_EVENT, this.onMochaReady);
        eventEmitter.on(constants_1.ENTRY_ADDED_EVENT, this.onEntryAdded);
        eventEmitter.on(constants_1.ENTRY_REMOVED_EVENT, this.onEntryRemoved);
    }
    Reporter.prototype.logInfo = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.options.quiet) {
            log.apply(void 0, args);
        }
    };
    Reporter.prototype.clearTerminal = function () {
        if (this.options.clearTerminal && this.options.interactive) {
            process.stdout.write(process.platform === 'win32' ? '\x1B[2J\x1B[0f' : '\x1B[2J\x1B[3J\x1B[H');
        }
    };
    Reporter.displayErrors = function (severity, errors) {
        var errorCount = errors.length;
        var message = severity === 'error'
            ? "Failed to compile with " + chalk_1["default"].red(errorCount + " " + severity + "(s)")
            : "Compiled with " + chalk_1["default"].yellow(errorCount + " " + severity + "(s)");
        var titleColor = severity === 'error' ? formatTitleError : formatTitleWarn;
        log(titleColor('WEBPACK'), message);
        errors.forEach(function (err) { return log(err); });
    };
    return Reporter;
}());
function testRunnerReporter(options) {
    new Reporter(options); // eslint-disable-line no-new
}
exports["default"] = testRunnerReporter;
//# sourceMappingURL=testRunnerReporter.js.map